;*************************************************************************
; Application for using three interleaved software PWM routines to control the
; brightness of a Siemens 3-color LED.  The app utilizes the FSR to swap the
; order of green/blue/red counters such that one increments every pass through 
; main, one every other pass and one every fourth pass.  Each of these count
; values is run through a sine lookup table to create a rolling rainbow pattern
; out of the LED.  Code also implements a manual switch override that toggles
; through each color and, when held, ramps up the individual brightness of that
; color.  In addition, the device will go into sleep mode if left untouched
; after 65K counts, waking up every ~2 seconds from the WDT timeout to flash the
; LED.  Relies heavily on the prlx.asm set of macros
;************************************************************************

	LIST    P = 12C508,R=DEC
	
#INCLUDE <P12C508.INC>          ; Include processor definition
	__CONFIG _IntRC_OSC&_WDT_ON&_CP_OFF&_MCLRE_OFF; Set up config bits
	ERRORLEVEL 1,-305	   	      ; Turn OFF message ID 305

	CBLOCK      0x10	;15 locations used
	Gcnt
	Rcnt
	Bcnt
	Count
	Ocount
	Gsin
	Rsin
	Bsin
	Test
  	manual:2
	btest
	Check
	tcnt
	dcy
	ENDC

	;GPIO,5 is wired to Vdd for LED
	
#Define  red  GPIO,0		
#Define  grn  GPIO,1		
#Define  blu  GPIO,4		
#Define  Button  GPIO,3
#Define  Pin  GPIO,2  ; just toggling output
#Define  TO  STATUS,4

	org	0x0
reset_address		
	goto	Start	    ; Jump to start of program. 

Sine	addwf PCL,F		;128 sample sine function
         DT 128,134,141,147,153,159,165,171,177,183,194
 	 DT 199,204,209,214,218,223,227,231,234,238,241
	 DT 244,246,249,250,252,254,255,255,255,255,255
	 DT 255,255,254,252,250,249,246,244,241,238,234
	 DT 231,227,223,218,214,209,204,199,194,188,183
	 DT 177,171,165,159,153,147,141,134,128,122,115
	 DT 109,103,97,91,85,79,73,68,62,57,52,47,42,38
	 DT 33,29,25,22,18,15,12,10,7,6,4,2,1,1,0,0,0
	 DT 1,1,2,4,6,7,10,12,15,18,22,25,29,33,38,42,47
	 DT 52,57,62,68,73,79,85,91,97,103,109,115,122,128

#INCLUDE <PRLX.ASM>            ;pseudo-parallax macro set
Fsrtst  incf FSR
	  cjbel FSR,0xF2,out	;value in FSR for Bcnt (use 0xD2 for 509)
	    movl FSR,Gcnt 	;reset to Gcnt so loop through three addresses
out    retlw 0

Pwm	clrwdt
	csae Gsin,Count
	 bcf grn		;conditionally set Green LED
	csae Rsin,Count
	 bcf red		;conditionally set Red LED
	csae Bsin,Count
	 bcf blu		;conditionally set Blue LED
	incfsz Count		;3.8mSec PWM loop
	 goto Pwm
	movlw b'11111011'
	movwf GPIO		;start with all lights off 
Pwmlp   csae Bsin,Count		;do again for blue
	 bcf blu		;conditionally set Blue LED
	goto $+1
	incfsz Count		;6.2mSec PWM loop
	 goto Pwmlp
	movlw b'11111011'       ;
	movwf GPIO		;start with all lights off
       retlw 0

threepwm  call Pwm		;25mSec
	  call Pwm
	  call Pwm
	  call Pwm
	 retlw 0

convsin clc
	rrf Rcnt,W		;because only 128 sample sine function
	call Sine
	movwf Rsin
	clc
	rrf Gcnt,W
	call Sine
	movwf Gsin
	clc 
	rrf Bcnt,W
	call Sine
	movwf Bsin 
	retlw 0


;
;********************************************************************
;               Test Program
;*********************************************************************

Start   movwf OSCCAL		;cal up oscillator
	movlw b'11101000'
	TRIS	GPIO		;pins GP4,2,1,0 are outputs
	movlw b'11111011'
	movwf GPIO		;start with all lights off 
	movl FSR,Gcnt
	movlw b'10001111'	;WDT on ~2 seconds
	OPTION			;use pullups ***use for manual button*
	movl btest,Gsin		;start with blue for manual mode
	clrf manual+1
	jb TO,main
	 clrf GPIO      	;else from WDT timeout, lights on
flash	 djnz manual,flash
	djnz manual+1,flash	;~180mSec flash
	 movl GPIO,0xFF
	 clrwdt
	 sleep			;sleep again
			
main   	jb Button,roll
 	 call threepwm          ;25mSec debounce
	jb Button,roll		;make sure not a glitch
next	 incf btest
	 cjbel btest,Bsin,run
	  movl btest,Gsin	;bracket between Gsin and Bsin
run	 mov FSR,btest
	 clrf INDF              ;start with color zeroed
 	 call threepwm          ;debounce switch ~100mSec, flash bright
	 call threepwm
	 call threepwm
	 call threepwm
cont	  decf INDF		;takes 4.5seconds to hit 0x00
	  snz
	   incf INDF              ;cap at 1
 	  clrf manual+1
 	  call Pwm
	  call Pwm		;13mSec
	  csbl INDF,0xF0	;for first 16 samples
	   call threepwm        ;slow ramp at first
	  jnb Button,cont	;until done with color setting
sit 	 call threepwm		;26mSec
	 jb Button,keepsit
	 call threepwm          ;25mSec debounce
	 jnb Button,next         ;then do next color
keepsit	djnz manual,sit		;else sit here for ~6.5 seconds
       djnz manual+1,sit	;sit here for 25 minutes
 	movlw b'11111011'       ;
	movwf GPIO		;start with all lights off
	sleep			;then sleep
		
roll	call threepwm  		;25mSec total rolling code routine
	incf Test
	andl Test,b'00000011'   ;so test is only 0-3 inclusive
	snz
	 incf INDF              ;only if Test = 0 (every fourth time)
	call Fsrtst
	incf INDF
	call Fsrtst
	btfss Test,0		;every other time
	 incf INDF
	call Fsrtst		;take it back to beginning
	call convsin
	djnz Ocount,main	;7 Seconds
	call Fsrtst		;stagger PWM loops on colors
				;so 21Sec repeat
	djnz manual+1,main	;25 minutes until sleep
	movlw b'11111011'       ;
	movwf GPIO		;start with all lights off
	sleep
	END
