// 1802 Simulator for Camel Forth
// Copyright (c) 2009 Harold Rabbie

/*
  Hacked by DW Schultz to support the UT71 monitor which requires:

  1) RCA multi-level I/O aka I/O groups
  2) 1854 UART
  3) uPD765 floppy disk controller

  Most of the code to support Forth has been removed.

  termios is used to reconfigure the console so that is mostly in raw
  mode. It is allowed to handle ^C which is the only way to terminate
  this simulation. (Other than looking up the PID and using kill.)

  Reading the CDP1854 data port is a blocking call since, so far, no program
  loops checking the status.
*/

// This is an instruction simulator for the RCA 1802 8-bit microprocessor
// Intended for use with the UT71 monitor program and MicroDOS
// Invocation:	1802sim [-i] [-r] PROGNAME
// -i Enable instruction-level tracing
// -C calculate UT71 ROM checksum (ROM value in parenthesis)
// -r simulate RNU by starting execution at 0x8000
// -v print version and compilation date
// -a filename  -- open file as simulated drive 0
// -b filename  -- open file as simulated drive 1
// -c filename  -- open file as simulated drive 2
// -d filename  -- open file as simulated drive 3

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <termios.h>
#include <signal.h>

#include "1802sim.h"
#include "fdc.h"               // NEC uPD765 floppy disk controller

#define VERSION "1.0"

boolean itrace = FALSE;		// Instruction-level tracing
boolean rnu = FALSE;            // Run utility flag
boolean crc = FALSE;            // calculate UT71 ROM checksum
int group = 0;                  // I/O group select

//************** RCA 1802 registers ****************
reg_t R [16];		// Pointer registers
#define RN R[N].w	// As 16-bits

byte P, X, opcode;	// Current PC, X-reg, instruction
#define PC R[P].w	// Current Program Counter
#define IX R[X].w	// Current Index Register

reg_t DFU;		// D and DF flag

#define D  DFU.b.lo	// 8-bit D register
#define DF DFU.b.hi	// LS bit is DF flag
#define DW DFU.w	// As 9 bits

byte sim_mem [64*1024];	// Simulated 64kB memory



//************** CamelForth register assignments *********

#define CODEPC  0
#define CODE 	R[0].w		// code word program counter
#define IP  	R[1].w		// Forth instruction pointer
#define PSP 	R[2].w		// Forth parameter stack pointer
#define RSP 	R[3].w		// Forth return stack pointer
#define COLONPC 5

#define DISABLED 1000		// disable forth-level tracing
word param_stack, return_stack, abort_addr, end_dict, exit_addr;
word breakpoint = 0;
int ftrace_level = DISABLED;	// nesting level of colon definitions
				// trace is printed when ftrace_level == 0
void print_ftrace (void);
int print_val (word addr);


#define makew(lo,hi) (((hi)<<8) | (lo))	/* make 16-bit word from two bytes */
#define TOS makew (mread (PSP), mread (PSP + 1))

#define mread(addr) (sim_mem [(addr)])	/* Read from simulated memory */

void cold_start (void) {
	P = X = 0;		// do a cold start
	CODE = rnu ? 0x8000 : 0;
}

void mwrite (word addr, byte data) {	// Write to simulated memory		
	if (addr >= 0x8000 && addr < 0x8800) {	//Check for write-protect violation
	  // printf ("Attempt to write to PROM\n");
	  // cold_start ();
	  return;
	}
	sim_mem [addr] = data;
}

//************** Opcode implementations *********

void unimplemented (void) {
	printf ("Unimplemented opcode at ");
	print_val (PC - 1);
	printf ("\r\n");
	cold_start();
}

void LDN (byte N) {
	if (N == 0) unimplemented ();		// IDL instruction
	D = mread (RN);
}

void INC (byte N) {
	RN++;
}

void DEC (byte N) {
	RN--;
}

void LDA (byte N) {
	D = mread (RN++);
}

void STR (byte N) {
	mwrite (RN, D);
}

void GLO (byte N) {
	D = R[N].b.lo;
}

void GHI (byte N) {
	D = R[N].b.hi;
}

void PLO (byte N) {
	R[N].b.lo = D;
}

void PHI (byte N) {
	R[N].b.hi= D;
}

void SEP (byte N) {
	P = N;
}

void SEX (byte N) {
	X = N;
}

#define FETCH mread(PC++)

#define SBR R[P].b.lo = ofs

void BRANCH (byte N) {		// 0x3N
	byte ofs = FETCH;

	switch (N) {
	case 0:	          SBR; break;	// BR
	case 2:	 if (!D)  SBR; break;	// BZ
	case 3:	 if (DF)  SBR; break;	// BDF
	case 8:	 	       break;	// SKP
	case 10: if (D)   SBR; break;	// BNZ
	case 11: if (!DF) SBR; break;	// BNF
	case 6:                         // B3
	  switch(group)
	    {
	    case 8:                     // NEC group -uPD765 FDC
	      if(fdc_int())
		SBR;
	      break;
	    }
	  break;
	case 14:                        // BN3
	  switch(group)
	    {
	    case 8:                     // NEC group - uPD765 FDC
	      if(!fdc_int())
		SBR;
	      break;
	    }
	  break;
	case 7:                         // B4
	  switch(group)
	    {
	    case 1:                     // RCA group, serial data in
	      SBR;                      // assume always true?
	      break;
	    }
	  break;
	case 15:                        // BN4
	  switch(group)
	    {
	    case 1:                     // RCA group, serial data in
	      break;
	    }
	  break;
	default: unimplemented ();	// 31 BQ, 34-37 Bn, 39 BNQ, 3C-3F BNn
	}
}

#define LBR { tempPC = PC; R[P].b.hi = mread (tempPC++); R[P].b.lo = mread (tempPC); }
#define LSKP (PC += 2)

void LBRANCH (byte N) {		// 0xCN
	word tempPC;

	switch (N) {
	case 0:	LBR; 				break;	// LBR
	case 2:	if (!D)	  LBR	else LSKP; 	break;	// LBZ
	case 3: if (DF)   LBR	else LSKP;	break;	// LBDF
	case 4: 				break;	// NOP
	case 6: if (D)    LSKP; 		break;	// LSNZ
	case 7: if (!DF)  LSKP; 		break;	// LNF
	case 8: LSKP; 				break;	// LSKP
	case 10: if (D)   LBR	else LSKP;	break;	// LBNZ
	case 11: if (!DF) LBR	else LSKP; 	break;	// LBNF
	case 14: if (!D)  LSKP; 		break;	// LSZ
	case 15: if (DF)  LSKP; 		break;	// LSDF
	default: unimplemented ();	// C1 LBQ, C5 LSNQ, C9 LBNQ, CC LSIE, CD LSQ
	}
}

void CODE60 (byte N) {			// 0x6N
	byte c;
	
	switch (N) {
	case 0:	IX++; break;	// IRX
	case 1: 			// OUT 1 - set I/O group
	  group = mread(IX++);
	  break;
	case 2:
	  switch(group)
	    {
	    case 1:                // RCA group - 1854 out char
	    c = mread(IX++);
	    putchar(c&0x7f);
	    break;
	    }
	  break;
	case 3:
	  switch(group)
	    {
	    case 1:               // RCA group - 1854 control word
	      c = mread(IX++);
	      break;
	    }
	  break;
	case 4:                  // OUT 4
	  switch(group)
	    {
	    case 8:              // NEC group - FDC control
	      fdc_control(mread(IX++));
	      break;
	    }
	  break;
	case 5:                  // OUT 5
	  switch(group)
	    {
	    case 8:              // NEC group - uPD765 command/data register
	      fdc_command(mread(IX++));
	      break;
	    }
	  break;
	case 7:                  // OUT 7
	  switch(group)
	    {
	    case 8:              // NEC group - uPD765 DMA count
	      fdc_set_dma(mread(IX++));
	      break;
	    }
	  break;
	case 9:	
	  break;
	case 10:	         // INP 2
	  switch(group)
	    {
	    case 1:              // RCA group - 1854 input character
	      c = getchar();     // block waiting for a character
	      mwrite(IX, c);
	      D = c;
	      break;
	    }
	  break;
	case 11:                 // INP 3
	  switch(group)
	    {
	    case 1:              // RCA group - 1854 status
	      c = 0x81;          // always ready
	      mwrite(IX, c);
	      D = c;
	      break;
	    }
	  break;
	case 12:                 // INP 4
	  switch(group)
	    {
	    case 8:              // NEC group - uPD765 status
	      c = fdc_mstatus();
	      mwrite(IX, c);
	      D = c;
	      break;
	    }
	  break;
	case 13:                 // INP 5
	  switch(group)
	    {
	    case 8:              // NEC group - uPD765 command/data
	      c = fdc_data();
	      mwrite(IX, c);
	      D = c;
	      break;
	    }
	  break;
	default: unimplemented();	// all others
	}
}

#define MOPR (mread (IX))		/* Memory operand */

void ALU_7N (byte N) {			// 0x7N
	byte DFS;

	switch (N) {
	case 0: exit (0);			break;	// RET - unimplemented
	case 1: DFS = mread(IX++);          // DIS
	  X = DFS >> 4;
	  P = DFS & 0x0f;
	  break;
	case 2: D = mread (IX++); 		break;	// LDXA
	case 3: mwrite (IX--, D); 		break;	// STXD
	case 4: DW = MOPR + D + DF; 		break;	// ADC
	case 5: DW = MOPR - D - !DF; DF = !DF; 	break;	// SDB
	case 6: DFS = D & 1; DW >>= 1; DF = DFS;	break;	// RSHR
	case 7: DW = D - MOPR - !DF; DF = !DF; 	break;	// SMB
	case 12: DW = FETCH + D + DF; 		break;	// ADCI
	case 13: DW = FETCH - D - !DF; DF = !DF;	break;	// SDBI
	case 14: DW = (DW << 1) | DF; DF &= 1;	break;	// RSHL
	case 15: DW = D - FETCH - !DF; DF = !DF;	break;	// SMBI
	default: unimplemented ();	
			// 71 DIS, 78 SAV, 79 MARK, 7A REQ, 7B SEQ
	}
}
	
void ALU_FN (byte N) {
	switch (N) {			// 0xFN
	case 0:	D = MOPR; 			break;	// LDX
	case 1:	D |= MOPR; 			break;	// OR
	case 2:	D &=  MOPR; 			break;	// AND
	case 3:	D ^= MOPR; 			break;	// XOR
	case 4:	DW = D + MOPR; 			break;	// ADD
	case 5:	DW = MOPR - D; DF = !DF; 	break;	// SD
	case 6:	DF = DW & 1; D >>=1; 		break;	// SHR
	case 7:	DW = D - MOPR; DF = !DF; 	break;	// SM
	case 8:	D = FETCH; 			break;	// LDI
	case 9:	D |= FETCH; 			break;	// ORI
	case 10: 	D &= FETCH; 		break;	// ANI
	case 11: 	D ^= FETCH; 		break;	// XRI
	case 12: 	DW = D + FETCH; 	break;	// ADI
	case 13: 	DW = FETCH - D; DF = !DF; 	break;	// SDI
	case 14: 	DW <<= 1; DF &= 1; 		break;	// SHL
	case 15: 	DW = D - FETCH; DF = !DF; 	break;	// SMI
	}
}

//	Dispatch table for bits [7:4] of opcode

typedef void (*opcode_t)(byte);

const opcode_t MSB_decode [16] =  {
	&LDN,		// 0N
	&INC,		// 1N
	&DEC,		// 2N
	&BRANCH,	// 3N
	&LDA,		// 4N
	&STR,		// 5N
	&CODE60,	// 6N
	&ALU_7N,	// 7N
	&GLO,		// 8N
	&GHI,		// 9N
	&PLO,		// AN
	&PHI,		// BN
	&LBRANCH,	// CN
	&SEP,		// DN
	&SEX,		// EN
	&ALU_FN		// FN
};

word read_hex_file (const char *file_name);
void read_symbols (const char *file_name);

//************** Disassembler for instruction-level tracing

typedef struct {
	char * MSB_disasm;
	char * LSB_disasm [16];
} disasm_t;

const disasm_t disassemble [16] = {
	{ "LDN %d" },
	{ "INC %d" },
	{ "DEC %d" },
	{ NULL, { "BR", "BQ", "BZ", "BDF", "B1", "B2", "B3", "B4", 
		  "SKP", "BNQ", "BNZ", "BNF", "BN1", "BN2", "BN3", "BN4" }},
	{ "LDA %d" },
	{ "STR %d" },
	//	{ "I/O %d" },
	{ NULL, { "IRX", "OUT 1", "OUT 2", "OUT 3", "OUT 4", "OUT 5", "OUT 6", "OUT 7",
		  "ILL", "INP 1", "INP 2", "INP 3", "INP 4", "INP 5", "INP 6", "INP 7" }},
	{ NULL, { "RET", "DIS", "LDXA", "STXD", "ADC", "SDB", "SHRC", "SMB", 
		  "SAV", "MARK", "REQ", "SEQ", "ADCI", "SDBI", "SHLC", "SMI" }},
	{ "GLO %d"},
	{ "GHI %d" },
	{ "PLO %d" },
	{ "PHI %d" },
	{ NULL, {"LBR", "LBQ", "LBZ", "LBDF", "NOP", "LSNQ", "LSNZ", "LSNF",
		"LSKP", "LBNQ", "LBNZ", "LBNF", "LSIE", "LSQ", "LSZ", "LSDF" }},
	{ "SEP %d" },
	{ "SEX %d" },
	{ NULL, { "LDX", "OR", "AND", "XOR", "ADD", "SD", "SHR", "SM", 
		  "LDI", "ORI", "ANI", "XRI", "ADI", "SDI", "SHL", "SMI" } }
};

//************** Print instruction-level trace

void print_itrace (void) {
	int len;

	byte opcode;
	const disasm_t * disasm;
	const char * str;

	len = print_val (PC);

	printf ("            " + len);

	opcode = mread (PC);
	disasm = &disassemble [(opcode >> 4) & 0xF];
	if (disasm->MSB_disasm) {
		str = disasm->MSB_disasm;
		printf (str, opcode & 0xF);
		len = strlen (str);
		if ((opcode & 0xF) < 10) len--;
	}
	else {
		str = disasm->LSB_disasm [opcode & 0xF];
		len = strlen (str);
		printf (str);
	}
	printf ("        " + len);
	printf ("DF:%d D:%02X [X]:%02X\r\n", DF, D, MOPR);

}

/*
  Restore terminal settings and close files before exiting.
 */
static struct termios oldattr;

void termination_handler(int signum)
{
  tcsetattr(STDIN_FILENO, TCSANOW, &oldattr);  // restore terminal settings
  fdc_termination();
  printf("\nsimulation terminated, PC = ");
  print_val(PC);
  printf("\n");
  exit(0);
}

/*
  UT71 has a self test that performs a simple checksum on the ROM contents.
  Since the assembler can't calculate the required value, this routine 
  does. It then prints the calculated value with the value in the ROM image
  in parenthesis.

  Patching the hex file is tedious since you have to find the correct record
  and then recompute its checksum.
 */

void chksum(void)
{
  int i, sum;

  for(sum = i = 0x8000; i < 0x87ff; i++)
    sum += mread(i);
  fprintf(stderr, "chksum byte =0x%02X (0x%02X)\r\n", 0x100-(sum&0xff), mread(0x87ff)&0xff);
}

//************** Read input files, interpret opcodes

int main (int argc, char *argv[])
{
  char * filename = NULL;
  int c;
  struct termios newattr;

  // Process command line options
  while((c = getopt(argc, argv, "iCrva:b:c:d:")) != -1)
    {
      switch(c)
	{
	case 'i': itrace = TRUE; break;
	case 'C': crc = TRUE; break;
	case 'r': rnu = TRUE; break;
	case 'v':
	  printf("Version %s, %s\n", VERSION, __DATE__);
	  exit(0);
	case 'a': fdc_open_image(0, optarg); break;
	case 'b': fdc_open_image(1, optarg); break;
	case 'c': fdc_open_image(2, optarg); break;
	case 'd': fdc_open_image(3, optarg); break;
	case '?':
	  fprintf(stderr, "Unknown option -%c\n", c);
	  exit(1);
	}
    }
  if(argc == optind)
    {
      fprintf (stderr, "Must provide target app file name\n");
      exit (1);
    }
  filename = argv[optind];

  
  P = X = 0;
  read_hex_file (filename);

  if(crc)
    chksum();
  
  R[0].w = rnu ? 0x8000 : 0;   // Run Utility?
  /*
    Install a handler for various termination events so that we have the
    opportunity to restore the terminal settings.
   */
  if (signal (SIGINT, termination_handler) == SIG_IGN)
    signal (SIGINT, SIG_IGN);
  if (signal (SIGHUP, termination_handler) == SIG_IGN)
    signal (SIGHUP, SIG_IGN);
  if (signal (SIGTERM, termination_handler) == SIG_IGN)
    signal (SIGTERM, SIG_IGN);

  /*
    Set the terminal to raw mode (no echo and not cooked) so that it looks
    like a dumb serial port.
   */
  tcgetattr(STDIN_FILENO, &oldattr);
  newattr = oldattr;
  cfmakeraw(&newattr);

  newattr.c_lflag |= ISIG;    // uncomment to process ^C

  newattr.c_cc[VMIN] = 1;       // block until at least one char available
  newattr.c_cc[VTIME] = 0;
  tcsetattr(STDIN_FILENO, TCSANOW, &newattr);

  printf ("Starting address: %4.4X\n", R[0].w);
  for (;;) {
    if (itrace) print_itrace ();
    opcode = FETCH;
    MSB_decode [(opcode >> 4) & 0xF] (opcode & 0xF);
  }
  termination_handler(0);
  return 0;
}

//************** Read Intel Hex .OBJ file

#define BUFSIZE 128
char buf [ BUFSIZE ];

void bad_record (const char *message) {
	printf ("Bad record : %s : %s\n", message, buf);
	exit (1);
}

word read_hex_file (const char *fnm) {	// returns starting address

    char *in_record_ptr;
    byte checksum;
    int count, out_address, type, temp, i;
    FILE * fd;

    if (!(fd = fopen( fnm, "r") )) {
        printf( "Can't open input file %s\n", fnm );
	exit (1);
    }

    for (;;) {
        if (!fgets( buf, BUFSIZE, fd)) break;
        in_record_ptr = buf;
        if (* in_record_ptr++ != ':') continue;

        if (sscanf( in_record_ptr, "%2x%4x%2x", &count, &out_address,
        	&type) != 3 ) bad_record( "Invalid count, address, type" );

        if (type) {
		fclose (fd);
		return out_address;      // type 1 is EOF
	}

        in_record_ptr += 8;
    	checksum = (out_address & 0xFF) + (out_address >>8) + count;

    	for (i = 0; i < count; i++) {
        	// read xsum field, too
        	if (sscanf( in_record_ptr, "%2x", &temp ) != 1 ) 
			bad_record( "Invalid hex data" );
		sim_mem [out_address++] = temp;
        	checksum += temp;
        	in_record_ptr += 2;
    	}
	if (sscanf( in_record_ptr, "%2x", &temp ) != 1 ) 
	  bad_record( "Invalid hex data" );
	checksum += temp;
    	if (checksum) bad_record( "Checksum error" );

    }
    printf ("Premature end of file %s\n", fnm);
    exit (1);
    return 0;
}       // end read_hex_file

	
// Look up address in symbol table and print symbolically
// Return length of string printed

int print_val (word addr) {
	printf ("%04X", addr);
	return 4;
}

	
